/*
 * giffer.c - tool to make animated gifs loop infinite (and stuff)
 * 
 * giffer <files>
 * -u, --update: if needed, update files and make the loop infinite
 * --read-only: do not modify the files; just display information
 * --read-comments: read & display comments & plain text blocks
 * --remove-comments: remove comments & plain text extension blocks
 * --remove-unknown: remove unknown application headers
 * --nuke: copy this file and put it in a new one, without comments,
 *         unknown application blocks, and other "useless" stuff.
 * --: treat every argument as a file from here
 * 
 * how to compile(probably):
 * cc -O3 -o giffer giffer.c -Wall
 * 
 * 
 * Made in 2017 by https://joppiesaus.neocities.org
 * 
 * Minor edit 2018: added missing free for o_name
 * 
 * another edit 2018: add support for emscripten
 * 
 * 
 * do whatever you want; please be careful using this program. Always
 * make backups of your GIF files before using this program. I made it
 * with a good intention; please no blamerino if something goes wrong
 * i cri already if it tortures your cat ;_; or something
 * 
 * Orrr if you don't understand that:
 * 
 * Permission to use, copy, modify, and/or distribute this software for
 * any purpose is hereby granted.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE
 * AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 * 
 */

#include <stdio.h> /* fgetc, fseek... so much more */
#include <string.h> /* strcmp, strcat, memcpy */
#include <stdlib.h> /* malloc, realloc */

#define DEBUG 0 /* Enable for a mess */
#define DISPLAY_SKIPS (DEBUG && 0)
#define VERBOSE (DEBUG || 1)

#define DELTA_ARR_INIT_SIZE 8

#ifdef __EMSCRIPTEN__
#include <emscripten.h>

#if DEBUG
EM_JS(int, get_total_memory, (), {
  return TOTAL_MEMORY;
});
#endif
#endif

/* reports progress to the javascript */
/* disable for nodejs and stuff */
#define EMS_REPORT (__EMSCRIPTEN__ && 1)
#if EMS_REPORT
#include <stdint.h>
#endif

struct g_delta
{
	size_t start;
	size_t length;
};

FILE * f;

/* TODO: Will probably go crazy on unexpected EOF (char c to int c)*/
char checkRange(const char * range)
{
	char c;
	for (size_t i = 0; range[i] != '\00'; i++)
	{
		c = fgetc(f);
		if (c != range[i]) return 0;
	}
	return 1;
}

char checkSwitch(const char * arg, size_t offs, const char * s)
{
	size_t j = 0;
	do
	{
		if (arg[offs] != s[j++])
		{
			return 0;
		}
	}
	while (arg[offs++] != '\00');
	return 1;
}

size_t calcColorTableSize(char c)
{
	/* first bit: global color table flag.
	 * if 1: global table is present */
	
	/* last 3 bits: color table size
	 * colors = 2^(n+1) where n = last 3 bits
	 * bytes = colors * 3 */
	 
	/* start with 2^1, multiply it by 3 to get the bytes instead
	 * of colors */
	size_t ct_size = 2 * 3;
	
	/* shifting to left is equivalent to multiplying by 2 */
	ct_size <<= (c & 0x07);
	return ct_size;
}

int main(int argc, char **argv)
{
	int c; /* current character read from file */
	unsigned short loops; /* amount of loops the GIF animation makes */
	char update = 0; /* update the loop count to infnity */
	char read_comments = 0;
	
	/* remove unknown application extensions(i.e. everything except
	 * NETSCAPE2.0 */
	char rm_unknown_appext = 0;
	char rm_comments = 0; /* remove comments & plain text extension block */
	/*char rm_mode = 0;  field to check if things are being removed
						(i.e. rm_comments or something else is true) */
	
	size_t i = 1; /* first arg is program name; skip */
	size_t n = 0; /* amount of files */
	
	#if __EMSCRIPTEN__ && DEBUG
	printf("Memory: %d\n", get_total_memory(0));
	#endif
	
	
	/* lmao this program will segfault w/o any args
	 * (i.e. not even program name) */
	char * files[argc - 1]; /* exclude first element(program name) */
	/*char ** files = malloc((argc - 1) * sizeof(char *));*/
	
	/* Argument processing */
	for (/*size_t i = 1*/; i < argc; i++)
	{
		/* TODO: Add unknown flags/switches not to files but
		 * display warning instead? */
		if (*argv[i] == '-')
		{
			if (argv[i][1] == '-')
			{
				if (argv[i][2] == '\00')
				{
					/* -- detected,
					 * disable argument processing from here */
					while (++i < argc)
					{
						/* Add the remaining args to the files */
						files[n++] = argv[i];
					}
					break;
				}
				
				/* switch */
				if (checkSwitch(argv[i], 2, "update"))
				{
					update = 1;
					continue;
				}
				if (checkSwitch(argv[i], 2, "read-only"))
				{
					update = 0;
					continue;
				}
				if (checkSwitch(argv[i], 2, "read-comments"))
				{
					read_comments = 1;
					continue;
				}
				if (checkSwitch(argv[i], 2, "nuke"))
				{
					rm_unknown_appext = 1;
					rm_comments = 1;
					continue;
				}
				if (checkSwitch(argv[i], 2, "remove-unknown"))
				{
					rm_unknown_appext = 1;
					continue;
				}
				if (checkSwitch(argv[i], 2, "remove-comments"))
				{
					rm_comments = 1;
					continue;
				}
			}
			else
			{
				/* flag */
				if (checkSwitch(argv[i], 1, "u"))
				{
					update = 1;
					continue;
				}
			}
		}
		
		/* It's a file(or an unkown switch); add it to the file list */
		files[n++] = argv[i];
	}
	
	if (n == 0)
	{
		/* no files specified */
		printf("Usage: %s <files>\n", *argv);
		return 0;
	}
	
	char rm_mode = rm_comments || rm_unknown_appext;
	
	/* ! delta_arr MEMORY IS UNINITIALIZED ! */
	struct g_delta * delta_arr = NULL;
	size_t delta_arr_size = DELTA_ARR_INIT_SIZE;
	size_t delta_i = 0;
	char rm_curr = 0;
	
	#if __EMSCRIPTEN__
	char this_updated;
	#endif
	
	if (rm_mode)
	{
		delta_arr = malloc(DELTA_ARR_INIT_SIZE * sizeof(struct g_delta));
	}
	
	/* TODO: Resize file array? */
	
	for (i = 0; i < n; i++)
	{
		#if __EMSCRIPTEN__
		this_updated = 0;
		#endif
		#if EMS_REPORT
		EM_ASM({
			giffer_fileBegin(UTF8ToString($0));
		}, files[i]);
		#endif
		
		printf("%zu: %s: ", i + 1, files[i]);
		fflush(stdout);
		f = fopen(files[i], update ? "r+" : "r");
		
		if (f == NULL)
		{
			perror(files[i]);
			continue;
		}
		
		#define GGCHAR (c = fgetc(f))
		
		/* Check if GIF (GIF header is either GIF89a or GIF87a */
		if (!checkRange("GIF8") ||
			(GGCHAR != '9' && c != '7') ||
			GGCHAR != 'a')
		{
			printf("not a GIF file\n");
			goto close;
		}
		
		/* Skip screen sizes(2+2=4 bytes) */
		fseek(f, 4, SEEK_CUR);
		GGCHAR; /* get "packed" field */
		if (c == EOF)
		{
			printf("unexpected EOF in GIF header\n");
			goto close;
		}
		
		fseek(f,
			(c & 0x80) /* global color table is present */
				? 2 + calcColorTableSize((char)c) /* skip it... */
				: 2, /* + rest of header */
			SEEK_CUR
		);
		
		/* Now we're past the header, and into the block zone 
		 * Usually, the loop header follow next in animated GIFs */
		 
		/* TODO: Research C optimalization tips */
		
		/* TODO: Weird behaviour EOF */
		do
		{
			GGCHAR;
			
			if (c == 0x21) /* Extension Block */
			{
				int label = fgetc(f);
				
				if (label == 0xfe)
				{
					if (rm_comments)
					{
						#if VERBOSE
						printf("removing a comment block\n");
						#endif
						rm_curr = 1;
						delta_arr[delta_i].start = ftell(f) - 2;
					}
					
					if (read_comments)
					{
						while (GGCHAR != 0x00)
						{
							if (c == EOF)
							{
								printf("unexpected EOF in comment block");
								goto close;
							}
							
							for (size_t j = 0; j < c; j++)
							{
								fputc(fgetc(f), stdout);
							}
						}
						fflush(stdout);
						goto rm_check;
					}
				}
				else
				{
					int blocksize = GGCHAR;
					
					if (label == 0xff) /* Application extension block */
					{
						/* 8 for the application identifier, 3 for 
						 * authentication code, 1 for null char, totals 12 */
						#define AEB_BUFSIZE 12
						
						char buf[AEB_BUFSIZE];
						
						if (fgets(buf, AEB_BUFSIZE, f) == NULL)
						{
							perror("fgets reading label");
							goto close;
						}
						
						#if DEBUG
						printf("APPHEADER %s\n", buf);
						#endif
						
						
						if (/*blocksize == 0x0b &&*/
							strcmp(buf, "NETSCAPE2.0") == 0)
						{
							loops = (char)GGCHAR;
							GGCHAR;
							/* loops is now used as a temporary variable
							 * to check if the loop header is intact */
							
							if (loops != 0x03 || c != 0x01)
							{
								/* Corrupted */
								printf("corrupted loop subblock\n");
								if (update)
								{
									printf("fixing...\n");
									fseek(f, -2, SEEK_CUR);
									fputc(0x03, f);
									fputc(0x01, f);									
									#if __EMSCRIPTEN__
									this_updated = 1;
									#endif
								}
							}
							
							/* Read the 2 byte unsigned short */
							GGCHAR;
							loops = (unsigned short)c << 0x08;
							GGCHAR;
							loops += (unsigned short)c;
							
							if (loops == 0x0000)
							{
								printf("loops infinite times!\n");	
							}
							else if (update)
							{
								/* infinitiefy */
								printf("loops %d times, updating...\n", loops);
								fseek(f, -2, SEEK_CUR); /* go back 2 bytes */
								/* write the short to 0(infinite times)*/
								fputc(0x00, f);
								fputc(0x00, f);
								
								#if __EMSCRIPTEN__
								this_updated = 1;
								#endif	
							}
							else
							{
								printf("loops %d times\n", loops);
							}
							#if EMS_REPORT								
							EM_ASM({
								giffer_loopcount(UTF8ToString($0), $1);
							}, files[i], ((int32_t)loops));
							#endif	
							
							/* goto close if needed */
							if (!rm_mode && !read_comments)
							{
								goto close;
							}
						}
						else if (rm_unknown_appext)
						{
							#if VERBOSE
							printf("removing unknown header %s\n", buf);
							#endif
							rm_curr = 1;	
							delta_arr[delta_i].start = ftell(f) - 
								(AEB_BUFSIZE - 1) - 3;
						}
					}
					else
					{
						if (blocksize == EOF)
						{
							printf("unexpected EOF in extension block\n");
							goto close;
						}
						
						fseek(f, blocksize, SEEK_CUR);
						
						#if DISPLAY_SKIPS
						printf("skipped %d bytes in extension block\n", 
							blocksize);
						#endif
						
						if (label == 0x01)
						{
							/* I've never seen a GIF with this! */
							/* Oh wait found one from the author: 
							 * http://www.olsenhome.com/gif/BOB_89A.GIF
							 */
							if (rm_comments)
							{
								#if VERBOSE
								printf("removing a plain text extension block\n");
								#endif
								rm_curr = 1;
								delta_arr[delta_i].start = ftell(f) - 3
									- blocksize;
							}
							
							if (read_comments)
							{
								/* ugh more somewhat duplicate code */						
								while (GGCHAR != 0x00)
								{
									if (c == EOF)
									{
										printf("unexpected EOF in plain text extension");
										goto close;
									}
									
									/*printf("Plain Text Extension: ");*/
									for (size_t j = 0; j < c; j++)
									{
										fputc(fgetc(f), stdout);
									}
								}
								fflush(stdout);
								goto rm_check;
							}
						}
					}
				}
				
				while (GGCHAR != 0x00)
				{
					if (c == EOF)
					{
						printf("unexpected EOF in extension block");
						goto close;
					}
					
					fseek(f, c, SEEK_CUR);
					
					#if DISPLAY_SKIPS
					printf("skipped %d bytes of extension block data\n", c);
					#endif
				}
				
rm_check:
				if (rm_curr)
				{
					rm_curr = 0;
					delta_arr[delta_i].length = ftell(f) - 
						delta_arr[delta_i].start;
					if (++delta_i >= delta_arr_size)
					{
						/*delta_arr_size += DELTA_ARR_INIT_SIZE;*/
						delta_arr_size *= 2;
						delta_arr = realloc(
							delta_arr, 
							delta_arr_size * sizeof(struct g_delta)
						);
					}
				}
			}
			else if (c == 0x2c) /* Image Block */
			{
				/* skip position & dimensions */
				fseek(f, 8, SEEK_CUR);
				
				GGCHAR; /* get packed bit thing */
				
				/* skip rest of header + optional color table */
				fseek(f, 
					(c & 0x80) /* local color table is present */
						? 1 + calcColorTableSize((char)c)
						: 1, /* 1 for skip lzw min code size */
					SEEK_CUR
				);
				
				/* LZW image data follows, skip all that */
				
				/* 1: get length of subblock */
				while (GGCHAR != 0x00)
				{
					if (c == EOF)
					{
						printf("unexpected EOF in image data\n");
						goto close;
					}
					
					/* 2: skip that length */
					fseek(f, c, SEEK_CUR);
					
					#if DISPLAY_SKIPS
					printf("skipped %d bytes of image data\n", c);
					#endif
				}
				/* if it was zero, then it means the end of the block
				 * was reached */
			}
		}
		while (c != EOF);
		
#if __EMSCRIPTEN__
/* shifting labels... dangerous! see comment below */
close:
#endif
		
		#if __EMSCRIPTEN__
		/* when emscripten enabled, we also want to output a _new.gif
		 * file when updated, so that it can be downloaded
		 * this is kind of a bodge, but it works */
		if (this_updated && delta_i == 0)
		{
			delta_arr[0].start = 0;
			delta_arr[0].length = 0;
			delta_i = 1;
		}
		#endif
		/* no need to check for rm_mode because that's always true
		 * when delta_i > 0 */
		if (delta_i > 0) 
		{
			#if DEBUG
			char * p = (char *)delta_arr;
			for (
				size_t j = 0;
				j < delta_arr_size * sizeof(struct g_delta);
				j++
			)
			{
				printf("%02x", p[j]);
			}
			printf("\n");
			for (size_t j = 0; j < delta_i; j++)
			{
				printf("%zu, %zu\n", delta_arr[j].start, 
					delta_arr[j].length);
			}
			#endif
			
			/* generate file name */
			
			/* get index of last dot */
			size_t di = strlen(files[i]) - 1;
			for (; di > 0; di--)
			{
				if (files[i][di] == '.')
				{
					break;
				}
			}
			
			if (di == 0)
			{
				di = strlen(files[i]);
			}
			
			/* I hope this works lol */
			const char * ext = "_new.gif";
			char * o_name = malloc(di + strlen(ext) + 1);
			memcpy(o_name, files[i], di);
			o_name[di] = '\00';
			strcat(o_name, ext);
			
			#if DEBUG
			printf("writing to: %s\n", o_name);
			#endif
			
			FILE * o = fopen(o_name, "w");
			
			if (o == NULL)
			{
				perror("fopen output file");
			}
			else
			{
				rewind(f);
				
				size_t j = 0;
				size_t k = 0;
				size_t bytes_saved = 0;
				while (1)
				{
					if (j == delta_arr[k].start)
					{
						#if DEBUG
						printf("doing %zu, %zu\n", delta_arr[k].start, 
							delta_arr[k].length);
						#endif
						
						fseek(f, delta_arr[k].length, SEEK_CUR);
						j += delta_arr[k].length;
						bytes_saved += delta_arr[k].length;
						
						if (++k == delta_i)
						{
							/* done here */
							break;
						}
					}
					else
					{
						GGCHAR;
						#if DEBUG
						/* shouldn't happen; if this happens then
						 * something is terribly wrong */
						if (c == EOF)
						{
							printf("EOF happened in output file");
							break;
						}
						#endif
						fputc(c, o);
						j++;
					}
				}
				while (GGCHAR != EOF)
				{
					fputc(c, o);
				}
				
				fclose(o);
				
				printf("%zu bytes removed, writing to %s\n",
					bytes_saved , o_name);
					
				#if EMS_REPORT
				/* notify webpage that the file has been done */
				/* TODO: Add bytes saved */
				EM_ASM({
					giffer_fileDone(UTF8ToString($0), UTF8ToString($1),
					$2);
				}, files[i], o_name, ((int32_t)bytes_saved));
				#endif
			}
			
			/* cleanup */
			delta_arr = realloc(
				delta_arr,
				DELTA_ARR_INIT_SIZE * sizeof(struct g_delta)
			);
			delta_arr_size = DELTA_ARR_INIT_SIZE;
			delta_i = 0;
			
			free(o_name);
		}
		#if VERBOSE
		else if (!read_comments)
		{
			/* TOODODODOOD */
			/* TODO: What if there weren't read any comments? */
			if (rm_mode)
			{
				printf("nothing to do here\n");
			}
			else
			{
				printf("no loop header found\n");
			}
		}
		#endif

#if !__EMSCRIPTEN__
close:
#endif
		#if DEBUG
		printf("closing %s\n", files[i]);
		#endif
		fclose(f);
	}
	
	return 0;
}
